import json
import logging
import os


logger = logging.getLogger()

def _filter_recommendations(recommendations, key_name, top_operation_type, top_resource_type):
    """Filter recommendations by operation/resource class and return filtered+sorted recommendations."""
    filtered_recommendations = []
    for recommendation in recommendations:
        filtered_recommendation = [
            k
            for (k, v) in recommendation["Classes"].items()
            if ((top_operation_type in v or v == ["*"]) and (top_resource_type == k or k == "*"))
        ]
        if filtered_recommendation:
            filtered_recommendations.append(recommendation)
    filtered_recommendations = sorted(filtered_recommendations, key=lambda k: k[key_name])
    return filtered_recommendations

def _get_recommendations(top_operation_type, top_resource_type):
    # Read recommendations file
    recommendations_file_path = os.path.join(os.getcwd(), "recommendations.json")
    with open(recommendations_file_path) as recommendations_file:
        all_recommendations = json.load(recommendations_file)
    
    # Filter recommendations by operation/resource type.
    recommendations = {}
    recommendations["runbook"] = _filter_recommendations(all_recommendations["Runbook"], "Title", top_operation_type, top_resource_type)
    recommendations["aws_documents"] = _filter_recommendations(all_recommendations["AWSDocuments"], "Title", top_operation_type, top_resource_type)
    recommendations["ssm_documents"] = _filter_recommendations(all_recommendations["SSMDocuments"], "DocumentName", top_operation_type, top_resource_type)
    return recommendations
    

def main(event, context):
    # Set log level
    logger.setLevel(logging.DEBUG)

    # Parse event and get ticket metadata
    ticket_id = event.get("TicketId")
    ticket_title = event.get("TicketTitle")
    ticket_description = event.get("TicketDescription")
    ticket_creation_time = event.get("TicketCreationTime")
    top_operation_type = event.get("TopOperationType")
    top_resource_type = event.get("TopResourceType")
    
    if not ticket_id:
        raise Exception("Ticket ID (TicketId) is not provided.")
    if not ticket_title:
        raise Exception("Ticket title (TicketTitle) is not provided.")
    if not ticket_description:
        raise Exception("Ticket description (TicketDescription) is not provided.")
    if not ticket_creation_time:
        raise Exception("Ticket creation time (TicketCreationTime) is not provided.")
    
    logger.info(f"Ticket ID: {ticket_id}")
    logger.debug(f"Ticket Title: {ticket_title}")
    logger.debug(f"Ticket Description: {ticket_description}")
    logger.debug(f"Ticket Creation Time: {ticket_creation_time}")
    logger.debug(f"Top operation type: {top_operation_type}")
    logger.debug(f"Top resource type: {top_resource_type}")

    # Execute your automation code
    if not top_operation_type or not top_resource_type:
        # Logic if operation or resource classification is not available due to low score.
        logger.info(f"Operation or resource is not classified. Operation: {top_operation_type}, Resource: {top_resource_type}")
        """
        Add your logic here.
        """
    else:
        # Logic if both operation/resource classifications are available.
        recommendations = _get_recommendations(top_operation_type, top_resource_type)
        logger.debug(f"Recommendations: {json.dumps(recommendations, indent=4)}")
        """
        Add your logic here. For example,
        1. Assign the ticket to the engineer based on resources. e.g. Assign EC2 tickets to engineer who is EC2 SME.
        2. Call SSM Document.
        3. Create a comment with recommended runbook/AWS documents to the ticket.
        4. Recommend runbook/documents to your customers, and auto resolve the ticket.
        """
    
    return